//////////////////////////////////////////////////////////////////////////////////////////////
// gui_sept_heimdall - Script used for heimdall GUI callback
//
// Based on Crom's work.
//--------------------------------------------------------------------------------------------
// Last modified by :  	Septirage		2025/09/13
//////////////////////////////////////////////////////////////////////////////////////////////


#include "sept_heimdall_include"

void DecrementRetries(string sVarName)
{
	int nTries = GetLocalInt(GetModule(), sVarName);
	if (nTries > 0) {
		SetLocalInt(GetModule(), sVarName, nTries - 1);
	}
}

int HeimdallLogin(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sPassword, string sRemember)
{
	int bRemember = FALSE;
	
	if(SEPT_HEIMDALL_ALLOW_REMEMBER)
		bRemember = StringToInt(sRemember);

	string sRetryVar = SEPT_HEIMDALL_RETRIES + sPlayerName + "@" + sIP;

	int nTries = GetLocalInt(GetModule(), sRetryVar);
	
	// refuse if too many failed attempts
	if (nTries >= SEPT_HEIMDALL_SECURITY_BADPASSWORD_RETRIES) {
		Heimdall_ShowErrorMessage(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, sPlayerName, sIP, sCDKey, iPrivileges, SEPT_HEIMDALL_MSG_LOGINBLOCKED);
		return XPMSGSRV_HEIMDALL_RET_WAIT;
	}
	
	if (Heimdall_CheckPasswordMatch(sPlayerName, sIP, sCDKey, iPrivileges, sPassword)) {
		// Login success

		// Reset retry count
		DeleteLocalInt(GetModule(), sRetryVar);

		if (bRemember) {
			Heimdall_RememberPlayer(sPlayerName, sIP, sCDKey, iPrivileges);
		}

		// Continue to module
		XPMsgServer_CloseGUIScreen(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME);
		return XPMSGSRV_HEIMDALL_RET_ALLOW;
	}

	// Login failed

	// Increment retries count for rate-limiting
	nTries += 1;
	SetLocalInt(GetModule(), sRetryVar, nTries);
	DelayCommand(SEPT_HEIMDALL_SECURITY_BADPASSWORD_COOLDOWN, DecrementRetries(sRetryVar));

	// Display login failed message
	Heimdall_ShowErrorMessage(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, sPlayerName, sIP, sCDKey, iPrivileges, SEPT_HEIMDALL_MSG_LOGINFAILED);
	return XPMSGSRV_HEIMDALL_RET_WAIT;
}


int HeimdallRegister(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sPassword, string sPasswordConfirm)
{
	// Stop if the two passwords are different
	if (sPassword != sPasswordConfirm) {
		Heimdall_ShowErrorMessage(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, sPlayerName, sIP, sCDKey, iPrivileges, SEPT_HEIMDALL_MSG_REGISTER_PASSWORDMISMATCH);
		return XPMSGSRV_HEIMDALL_RET_WAIT;
	}

	string sErrMsg;

	// Stop if the password is not allowed (too short, not random enough, ...)
	sErrMsg = Heimdall_CheckPasswordPolicy(sPlayerName, sIP, sCDKey, iPrivileges, sPassword);
	if (sErrMsg != "") {
		Heimdall_ShowErrorMessage(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, sPlayerName, sIP, sCDKey, iPrivileges, sErrMsg);
		return XPMSGSRV_HEIMDALL_RET_WAIT;
	}

	// Try to register the account
	sErrMsg = Heimdall_RegisterNewAccount(sPlayerName, sIP, sCDKey, iPrivileges, sPassword);
	if (sErrMsg != "") {
		Heimdall_ShowErrorMessage(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, sPlayerName, sIP, sCDKey, iPrivileges, sErrMsg);
		return XPMSGSRV_HEIMDALL_RET_WAIT;
	}

	// Successfully registered account, print success message and close registration UI
	if (SEPT_HEIMDALL_MSG_REGISTER_SUCCESS != "")
		Heimdall_OpenMsgGUI(iUniquePlayerID, sPlayerName, sIP, sCDKey, iPrivileges, SEPT_HEIMDALL_MSG_REGISTER_SUCCESS);

	XPMsgServer_CloseGUIScreen(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME);
	return XPMSGSRV_HEIMDALL_RET_ALLOW;
}

int StartingConditional(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sAction, string sParam1, string sParam2)
{
	object oModule = GetModule();
	if(OBJECT_SELF != oModule)
	{
		WriteTimestampedLogEntry("Tentative frauduleuse d'utiliser le script GUI par '"+GetName(OBJECT_SELF)+"'. Nom de compte:'"+GetPCPlayerName(OBJECT_SELF)+"'. IP:+"+GetPCIPAddress(OBJECT_SELF));
		return XPMSGSRV_HEIMDALL_RET_WAIT;
	}

	if(sAction == "login")
	{
		return HeimdallLogin(iUniquePlayerID, sPlayerName, sIP, sCDKey, iPrivileges, sParam1, sParam2);
	}
	else if(sAction == "register")
	{
		return HeimdallRegister(iUniquePlayerID, sPlayerName, sIP, sCDKey, iPrivileges, sParam1, sParam2);
	}
	
	return XPMSGSRV_HEIMDALL_RET_WAIT;
}