//////////////////////////////////////////////////////////////////////////////////////////////
// sept_heimdall_config - Constants and functions to adapt to your needs
//
// Based on Crom's work.
//--------------------------------------------------------------------------------------------
// Last modified by :  	Septirage		2025/09/13
//////////////////////////////////////////////////////////////////////////////////////////////


#include "nwnx_sql"
#include "nwnx_msgserver"

//==============================================================================
//                                  BEHAVIOR
//==============================================================================

/// Set to TRUE to allow account registration when a player connects with an
/// unknown account
const int SEPT_HEIMDALL_ALLOW_REGISTRATION = TRUE;

/// Set to TRUE to show a remember me checkbox to future login on the same IP + Account
const int SEPT_HEIMDALL_ALLOW_REMEMBER = TRUE;

/// Number of successive bad passwords during login before the IP/account is rate-limited.
const int SEPT_HEIMDALL_SECURITY_BADPASSWORD_RETRIES = 4;

/// Duration of the rate-limit when too many bad passwords have been entered
/// See also SEPT_HEIMDALL_MSG_LOGINBLOCKED
const float SEPT_HEIMDALL_SECURITY_BADPASSWORD_COOLDOWN = 1800.0;


const string SEPT_HEIMDALL_DBNAME_ACCOUNTS = "sept_heimdall_accounts";
const string SEPT_HEIMDALL_DBNAME_REMEMBERED = "sept_heimdall_remembered";

//==============================================================================
//                               LOCALIZATION
//==============================================================================

/// Server name
const string SEPT_HEIMDALL_SERVERNAME = "PW Server Name";

/// Error message when a an account already exists but with another case
/// (can cause issues with scripts and database storage)
///
/// Tokens:
/// {{CORRECT_ACCOUNT}} Replaced with the correct account name
const string SEPT_HEIMDALL_MSG_KICK_BADCASE = "Your account case is incorrect. Please reconnect to the server using this account: '{{CORRECT_ACCOUNT}}'";

/// Error message when the player account name is unknown and regitration is
/// disabled (SEPT_HEIMDALL_ALLOW_REGISTRATION == FALSE)
const string SEPT_HEIMDALL_MSG_KICK_NEEDREGISTRATION = "You need to register an account before connecting to this server. Go to https://example.com to register yourself !";

/// Error message when the player has entered the wrong password
const string SEPT_HEIMDALL_MSG_LOGINFAILED = "Your account or password is incorrect";

/// Error message when a player has tried to login with too many wrong
/// passwords and has been rate limited. The rate limit duration is defined
/// by SEPT_HEIMDALL_SECURITY_BADPASSWORD_COOLDOWN
const string SEPT_HEIMDALL_MSG_LOGINBLOCKED = "You have tried too many bad accounts or passwords. Please retry in 30 minutes.";

/// Error message when during registration the player has entered different passwords
const string SEPT_HEIMDALL_MSG_REGISTER_PASSWORDMISMATCH = "The two passwords do not match";

/// Optional message displayed when a player registered their account before
/// selecting their character. Set to "" to disable
const string SEPT_HEIMDALL_MSG_REGISTER_SUCCESS = "Account registered !";

//==============================================================================
//                           DATABASE INTERACTIONS
//==============================================================================

/// Called only once per module boot, to setup the database.
void Heimdall_ModuleInit()
{
	XPMsgServer_SetAuthorizedGUIScript("gui_sept_heimdall");

	SQLExecDirect(
	    "CREATE TABLE IF NOT EXISTS `"+SEPT_HEIMDALL_DBNAME_ACCOUNTS+"` ("
	    + "  `account_name` varchar(32) COLLATE utf8mb4_bin NOT NULL,"
	    + "  `password_hash` varchar(512) COLLATE utf8mb4_bin DEFAULT NULL,"
	    + "  `password_salt` varchar(32) COLLATE utf8mb4_bin DEFAULT NULL,"
	    + "  PRIMARY KEY (`account_name`)"
	    + ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin;");
	SQLExecDirect(
	    "CREATE TABLE IF NOT EXISTS `"+SEPT_HEIMDALL_DBNAME_REMEMBERED+"` ("
	    + "  `account_name` varchar(32) COLLATE utf8mb4_bin NOT NULL,"
	    + "  `ip` varchar(15) COLLATE utf8mb4_bin NOT NULL,"
//with EE, cdkey no longer make much sense. Without it, you can want to store them
//	    + "  `cdkey` varchar(16) COLLATE utf8mb4_bin NOT NULL,"
//	    + "  PRIMARY KEY (`account_name`,`ip`,`cdkey`),"
	    + "  PRIMARY KEY (`account_name`,`ip`),"
	    + "  KEY `fk_pw_auth` (`account_name`),"
	    + "  CONSTRAINT `fk_pw_auth` FOREIGN KEY (`account_name`) REFERENCES `"+SEPT_HEIMDALL_DBNAME_ACCOUNTS+"` (`account_name`) ON DELETE CASCADE ON UPDATE CASCADE"
	    + ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin;");
}

/// Returns TRUE if the account is already registered and has been remembered
/// on a previous login
int Heimdall_IsRemembered(string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	string sPlayerNameEnc = SQLEncodeSpecialChars(sPlayerName);
	SQLExecDirect("SELECT account_name FROM `"+SEPT_HEIMDALL_DBNAME_REMEMBERED+"` WHERE "
	+ "account_name='" + sPlayerNameEnc + "'"
	+ " AND ip='" + sIP + "'"
//with EE, cdkey no longer make much sense. Without it, test them if you want
//	+ " AND cdkey='" + sCDKey + "'"
	);

	return SQLFetch();
}

/// Returns the correct account name for a player if they are registered, or an empty string if the player is not registered.
/// This function should be used for checking if a player is registered or if there is an account case issue
string Heimdall_GetRegisteredAccount(string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	string sPlayerNameEnc = SQLEncodeSpecialChars(sPlayerName);

	SQLExecDirect("SELECT account_name FROM `"+SEPT_HEIMDALL_DBNAME_ACCOUNTS+"` WHERE LOWER(account_name)=LOWER('" + sPlayerNameEnc + "')");
	if (!SQLFetch()) {
		// Account is not registered
		return "";
	}
	// Return stored account (for case check)
	return SQLGetData(1);
}

/// Returns a message if the player is not allowed to login to his account. Returns "" if the player is allowed to login.
/// This function can be used for refusing banned players, or imposing IP or time of day restrictions
/// Note: The result of this function is also checked when a remembered player enters the server
string Heimdall_IsLoginAllowed(string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	return "";
}



// Obfuscates the password to avoid having it in cleartext in SQL logs.
// This is NOT a real protection: the password could still be recovered from logs.
// Best: disable INFO-level parameter logging in the SQL plugin.
string ObfuscatePasswordToHex(string sPwd) {

	// "SECRET" pepper. Change to your own.
	string sPepper = "MySuperSecret";
	

    int nPwdLen = GetStringLength(sPwd);
    int nPepLen = GetStringLength(sPepper);
    if (nPepLen == 0) return "";
    string sOut = "";
    int i;
    for (i = 0; i < nPwdLen; i++) {
        int bPwd = CharToASCII(GetSubString(sPwd, i, 1)) & 0xFF;
        int bPep = CharToASCII(GetSubString(sPepper, i % nPepLen, 1)) & 0xFF;
        int bXor = bPwd ^ bPep;
        string h = IntToHexString(bXor);
        if (GetStringLength(h) < 2) h = "0" + h;
        sOut += h;
    }
    return sOut;
}

string Heimdall_RegisterNewAccount(string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sPassword)
{
	string sPlayerNameEnc = SQLEncodeSpecialChars(sPlayerName);
	string sPasswordEnc = SQLEncodeSpecialChars(sPassword);
	
	sPasswordEnc = ObfuscatePasswordToHex(sPasswordEnc);

	// Register account with password
	// Player password is never stored in db. The password must be salted then hashed to check that passwords matches without storing the password.
	SQLExecDirect("INSERT INTO `"+SEPT_HEIMDALL_DBNAME_ACCOUNTS+"` (account_name, password_salt, password_hash) VALUES ('" + sPlayerNameEnc + "', MD5(RAND()), SHA2(CONCAT('" + sPasswordEnc + "',password_salt), 512))");

	// Check that the account has been inserted
	if (SQLGetAffectedRows() == 1) {
		return "";
	} else {
		// This can happen when the SQL query fails
		return "Registration failed. Please contact an admin";
	}
}

int Heimdall_CheckPasswordMatch(string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sPassword)
{
	string sPlayerNameEnc = SQLEncodeSpecialChars(sPlayerName);
	string sPasswordEnc = SQLEncodeSpecialChars(sPassword);
	
	sPasswordEnc = ObfuscatePasswordToHex(sPasswordEnc);

	SQLExecDirect("SELECT SHA2(CONCAT('" + sPasswordEnc + "',`password_salt`), 512)=password_hash as `success` FROM `"+SEPT_HEIMDALL_DBNAME_ACCOUNTS+"` WHERE account_name='" + sPlayerNameEnc + "'");

	return SQLFetch() && StringToInt(SQLGetData(1)) == 1;
}

void Heimdall_RememberPlayer(string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	// Remember account+ip combination
	string sPlayerNameEnc = SQLEncodeSpecialChars(sPlayerName);
	SQLExecDirect("INSERT INTO `"+SEPT_HEIMDALL_DBNAME_REMEMBERED+"`"
//	+ " (account_name, ip, cdkey) VALUES ('" + sPlayerNameEnc + "', '" + sIP + "', '" + sCDKey + "')");
	+ " (account_name, ip) VALUES ('" + sPlayerNameEnc + "', '" + sIP + "')");
}

//==============================================================================
//                         ACCOUNT AND PASSWORD POLICY
//==============================================================================

/// Checks if a player is allowed to register the account sPlayerName
///
/// Returns an empty string if the account is acceptable, and an error message otherwise.
string Heimdall_CheckAccountPolicy(string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	int bValidAccount = TRUE;

	int nAccountLen = GetStringLength(sPlayerName);

	if (nAccountLen < 3) {
		return "Your account name must be longer than 3 characters";
	}
	if (nAccountLen > 32) {
		// Adjust max length to fit in the SQL tables
		return "Your account name cannot be longer than 32 characters";
	}
	// if(sCDKey == "01234567"){
	// 	return "This CDKey is not allowed on this server";
	// }
	return "";
}

/// Checks that the player password follows the server rules for secure passwords
///
/// Returns an empty string if the password is valid, and an error message otherwise.
string Heimdall_CheckPasswordPolicy(string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sPassword) //TODO
{
	int bValidPassword = TRUE;

	string sPolicy = "Your password must be at least 8 characters, with at least 1 lower case, 1 upper case and 1 number\nYour password has the following issues:\n";

	int nPassLen = GetStringLength(sPassword);
	int nCountLower = 0;
	int nCountUpper = 0;
	int nCountNum = 0;
	int nCountSpec = 0;
	int i;
	for (i = 0; i < nPassLen; i++) {
		int nChar = CharToASCII(GetSubString(sPassword, i, 1));
		if (nChar >= 0x30 && nChar <= 0x39)
			nCountNum++;
		else if (nChar >= 0x41 && nChar <= 0x5A)
			nCountUpper++;
		else if (nChar >= 0x61 && nChar <= 0x7A)
			nCountLower++;
		else
			nCountSpec++;
	}

	// At least 8 characters
	if (nPassLen < 8) {
		sPolicy += "Too short; ";
		bValidPassword = FALSE;
	}
	if (nCountLower < 1) {
		sPolicy += "Not enough lower-case characters; ";
		bValidPassword = FALSE;
	}
	if (nCountUpper < 1) {
		sPolicy += "Not enough upper-case characters; ";
		bValidPassword = FALSE;
	}
	if (nCountNum < 1) {
		sPolicy += "Not enough numbers;";
		bValidPassword = FALSE;
	}
	// if(nCountSpec < 1){
	// 	sPolicy += "Not enough special characters ";
	// 	bValidPassword = FALSE;
	// }

	if (bValidPassword) {
		return "";
	}
	return sPolicy;
}

//==============================================================================
//                               GUI CUSTOMIZATION
//==============================================================================

// Custom message box
void Heimdall_OpenMsgGUI(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sMessage)
{
	XPMsgServer_DisplayMessageInfo(iUniquePlayerID, "SCREEN_MESSAGEBOX_DEFAULT", sMessage, "OK");
}

void Heimdall_OpenKickGui(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges, string sMsg)
{
	XPMsgServer_DisplayGuiScreen(iUniquePlayerID, "SCREEN_SEPT_HEIMKICK", "death_default.xml");
	XPMsgServer_SetGUIObjectHidden(iUniquePlayerID, "SCREEN_SEPT_HEIMKICK", "BUTTON_RESPAWN", TRUE);
	XPMsgServer_SetGUIObjectText(iUniquePlayerID, "SCREEN_SEPT_HEIMKICK", "MESSAGE_TEXT", sMsg);
}

// Custom login GUI
const string SEPT_HEIMDALL_LOGIN_SCENENAME = "SCREEN_PW_AUTH_LOGIN";
void Heimdall_OpenLoginGUI(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	XPMsgServer_DisplayGuiScreen(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, "sept_heimdall_login.xml");

	XPMsgServer_SetGUIObjectText(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, "SERVERNAME", SEPT_HEIMDALL_SERVERNAME);
	XPMsgServer_SetGUIObjectText(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, "PlayerName", sPlayerName);
	if (!SEPT_HEIMDALL_ALLOW_REMEMBER) {
		XPMsgServer_SetGUIObjectHidden(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, "RememberMeCB", TRUE);
		XPMsgServer_SetGUIObjectHidden(iUniquePlayerID, SEPT_HEIMDALL_LOGIN_SCENENAME, "RememberMeTX", TRUE);
	}
}

// Custom registration GUI
const string SEPT_HEIMDALL_REGISTER_SCENENAME = "SCREEN_SEPT_HEIMDALL_REGISTER";
void Heimdall_OpenRegisterGUI(int iUniquePlayerID, string sPlayerName, string sIP, string sCDKey, int iPrivileges)
{
	XPMsgServer_DisplayGuiScreen(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, "sept_heimdall_register.xml");

	XPMsgServer_SetGUIObjectText(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, "SERVERNAME", SEPT_HEIMDALL_SERVERNAME);
	XPMsgServer_SetGUIObjectText(iUniquePlayerID, SEPT_HEIMDALL_REGISTER_SCENENAME, "PlayerName", sPlayerName);
}

void Heimdall_ShowErrorMessage(int iUniquePlayerID, string SCENENAME, string sPlayer, string sIP, string sCDKey, int iPrivileges, string sMsg)
{
	XPMsgServer_SetGUIObjectText(iUniquePlayerID, SCENENAME, "ErrorMsg", sMsg);
}